<?php
/**
 * Moderation Auto-PM
 * Automaticly sends a PM when a thread, post or attachment is moderated
 */

/**
 * Parses the message phrase replacing the {placeholder}s
 * 
 * @package 	mod_autopm
 * @version 	2.0.1
 * @author 		Alan Wagstaff
 * @copyright 	(C) 2006 Alan Wagstaff
 */

class autopm_parser
{

	/**
	 * An array of valid placeholders to parse
	 * 
	 * @var 	array
	 */
	var $placeholders = array('{user_id}',
							  '{user_name}',
							  '{moderator_id}',
							  '{moderator_name}',
							  '{item_id}',
							  '{item_name}',
							  '{source_id}',
							  '{source_name}',
							  '{dest_id}',
							  '{dest_name}',
							  '{reason}',
							  '{homepage_url}',
							  '{forum_url}',
							  '{old_title}'
							  );

	/**
	 * The original phrase text
	 * 
	 * @var 	char
	 */
	var $phrasetext = '';

	/**
	 * The parsed phrase text
	 * 
	 * @var 	char
	 */
	var $phrasetext_parsed = '';

	/**
	 * The array of text that will be parsed in to the phrase
	 * 
	 * @var 	array
	 */
	var $items = array();

	/**
	 * The item ID
	 * 
	 * @var 	int
	 */
	var $item_id = 0;

	/**
	 * The type of item (thread, post or attachment)
	 * 
	 * @var		char
	 */
	var $item_type = '';

	/**
	 * The action being performed
	 * 
	 * @var 	char
	 */
	var $action = '';

	/**
	 * Source ID (for thread or post moves)
	 * 
	 * @var 	int
	 */
	var $source = 0;

	/**
	 * Destination ID (for thread or post moves)
	 * 
	 * @var 	int
	 */
	var $dest = 0;

	/**
	 * Reason (for deletions)
	 * 
	 * @var 	char
	 */
	var $reason = '';

	/**
	 * Any extra text (new thread title, etc)
	 * 
	 * @var 	char
	 */
	var $extra = '';

	/**
	 * Can user view the source forum?
	 * 
	 * @var 	bool
	 */
	var $view_source = true;

	/**
	 * Can user view the destination forum?
	 * 
	 * @var 	bool
	 */
	var $view_dest = true;

	/**
	 * Recipients username
	 * 
	 * @var 	char
	 */
	var $recipient = '';

	/**
	 * A reference to the vBulletin object
	 * 
	 * @var 	object
	 */
	var $vbulletin = null;

	/**
	 * Constructor - sets up the vBulletin object
	 *
	 * @param 	object	The vBulletin object
	 *
	 * @return 	void
	 */
	function autopm_parser(&$vbobj)
	{

		$this->vbulletin =& $vbobj;

	}

	/**
	 * Parses the phrase
	 * 
	 * @return 		bool
	 */
	function parse_phrase()
	{

		if (strlen($this->phrasetext) < 1)
		{
			// No phrase text
			return false;
		}

		if (count($this->items) < 1)
		{
			// No args, just throw back the unparsed phrase then give up
			$this->phrasetext_parsed = $this->phrasetext;

			return true;
		}

		foreach ($this->items AS $id => $text)
		{
			// Loop through the args replacing each one we find in the phrase text
			if (in_array($id, $this->placeholders))
			{
				$this->phrasetext = str_replace($id, $text, $this->phrasetext);
			}
		}

		// Set the parsed text
		$this->phrasetext_parsed = $this->phrasetext;

		return true;

	}

	/**
	 * Fetches the vars for this->items
	 * 
	 * @return 		bool
	 */
	function fetch_items()
	{

		// 1. Fetch the item info depending on this->type
		switch ($this->item_type)
		{

			case 'thread':
				{

					$item_info_a = $this->vbulletin->db->query_first("SELECT title AS item_name, postuserid AS user_id, postusername AS user_name, dateline
						FROM " . TABLE_PREFIX . "thread
						WHERE threadid = " . $this->vbulletin->input->clean($this->item_id, TYPE_UINT));

					break;

				}

			case 'post':
			case 'attachment':
				{

					$item_info_a = $this->vbulletin->db->query_first("SELECT SUBSTRING(post.pagetext, 1, 60) AS item_name, post.userid AS user_id, post.username AS user_name,
							thread.threadid AS source_id, thread.title AS source_name, post.dateline AS dateline
						FROM " . TABLE_PREFIX . "post AS post
						LEFT JOIN " . TABLE_PREFIX . "thread AS thread ON (post.threadid = thread.threadid)
						WHERE postid=" . $this->vbulletin->input->clean($this->item_id, TYPE_UINT));

					break;

				}

		}

		// 2.  Check the user can view the source/dest forums, if so, get the forum names
		if ($this->item_type == 'thread')
		{

			$this->canview($item_info_a['user_id'], $this->source, $this->dest);

			// Source
			if ($this->view_source)
			{
				$source_name_a = $this->vbulletin->db->query_first("SELECT title_clean FROM " . TABLE_PREFIX . "forum WHERE forumid=" . $this->vbulletin->input->clean($this->source, TYPE_UINT));
			}

			// Destination
			if ($this->view_dest)
			{
				$dest_name_a = $this->vbulletin->db->query_first("SELECT title_clean FROM " . TABLE_PREFIX . "forum WHERE forumid=" . $this->vbulletin->input->clean($this->dest, TYPE_UINT));
			}

		}

		if ($this->item_type == 'post' AND $this->dest)
		{
			// Fetch the destination thread details for the post move
			$dest_name_a = $this->vbulletin->db->query_first("SELECT title AS title_clean FROM " . TABLE_PREFIX . "thread WHERE threadid=" . $this->vbulletin->input->clean($this->dest, TYPE_UINT));
		}

		if ($this->item_type == 'thread' AND $this->action == 'edit_title')
		{
			// Stuff new title in item_name and put the old title in title_old
			$title_old = $item_info_a['item_name'];
			$item_info_a['item_name'] = $this->extra;
		}

		// Check the age cutoff
		if ($this->vbulletin->options['cit_autopm_agecutoff'])
		{
			$cutoff = TIMENOW - $this->vbulletin->options['cit_autopm_agecutoff'] * 86400;

			if ($item_info_a['dateline'] < $cutoff)
			{
				return false;
			}
		}

		// Un-htmlspecialchars_uni
		$item_info_a['item_name'] = str_replace('&lt;', '<', $item_info_a['item_name']);
		$item_info_a['item_name'] = str_replace('&gt;', '>', $item_info_a['item_name']);
		$item_info_a['item_name'] = str_replace('&quot;', '"', $item_info_a['item_name']);

		// 3.  Fill in the blanks
		$this->set_items(array(
			'{user_id}'			=> $item_info_a['user_id'],
			'{user_name}'		=> $item_info_a['user_name'],
			'{moderator_id}'	=> $this->vbulletin->userinfo['userid'],
			'{moderator_name}'	=> $this->vbulletin->userinfo['username'],
			'{item_id}'			=> $this->item_id,
			'{item_name}'		=> $item_info_a['item_name'] . ($this->item_type == 'post' ? '...' : ''),
			'{source_id}'		=> ($this->item_type == 'thread' ? $this->source : $item_info_a['source_id']),
			'{source_name}'		=> ($this->item_type == 'thread' ? $source_name_a['title_clean'] : $item_info_a['source_name']),
			'{dest_id}'			=> $this->dest,
			'{dest_name}'		=> $dest_name_a['title_clean'],
			'{reason}'			=> $this->reason,
			'{homepage_url}'	=> $this->vbulletin->options['homeurl'],
			'{forum_url}'		=> $this->vbulletin->options['bburl'],
			'{old_title}'		=> $title_old,
			));

		// 4. Set the recipients for later use
		$this->recipient = $item_info_a['user_name'];

		return true;

	}

	/**
	 * Checks to see if the user has access to view the source/dest forum
	 * 
	 * @param 		int			The user ID to check
	 * @param 		int			The ID of the source forum
	 * @param 		int			The ID of the dest forum
	 * 
	 * @return 		void
	 */
	function canview($userid = 0, $source = 0, $dest = 0)
	{

		$userinfo = fetch_userinfo($userid);
		cache_permissions($userinfo);

		$this->view_source = ($userinfo['forumpermissions']["$source"] & $this->vbulletin->bf_ugp_forumpermissions['canview']);
		$this->view_dest = ($userinfo['forumpermissions']["$dest"] & $this->vbulletin->bf_ugp_forumpermissions['canview']);

	}

	/**
	 * Checks for a recycle bin forum
	 * 
	 * @return 		void
	 */
	function check_recycle_bin()
	{

		if (($this->dest == $this->vbulletin->options['cit_autopm_recyclebin']) AND ($this->item_type == 'thread') AND ($this->action == 'move'))
		{
			$this->action = 'delete';
		}

	}

	/**
	 * Cleans this->items and puts them in the array
	 * 
	 * @param 		array		An array of items to be parsed-in
	 * 
	 * @return 		bool
	 */
	function set_items($items_a)
	{

		// Check we've got something to do
		if ((!is_array($items_a)) OR (count($items_a) < 1))
		{
			return false;
		}

		// Clean the array and put it in this->items
		$this->items = $this->vbulletin->input->clean($items_a, TYPE_ARRAY_STR);

		return true;

	}

	/**
	 * Returns the parsed phrase text
	 * 
	 * @return 		char
	 */
	function get_phrasetext_parsed()
	{

		return $this->vbulletin->input->clean($this->phrasetext_parsed, TYPE_STR);

	}

}

?>
